﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace DuplicateFinder
{
    class Program
    {
        // Listing 11-10. Dodanie opcji /test.
        static void Main(string[] args)
        {
            bool recurseIntoSubdirectories = false;

            if (args.Length < 1)
            {
                ShowUsage();
                return;
            }

            int firstDirectoryIndex = 0;
            IEnumerable<string> directoriesToSearch = null;
            bool testDirectoriesMade = false;

            try
            {
                // Sprawdzamy, czy program działa w trybie testowym.
                if (args.Length == 1 && args[0] == "/test")
                {
                    directoriesToSearch = MakeTestDirectories();
                    testDirectoriesMade = true;
                    recurseIntoSubdirectories = true;
                }
                else
                {
                    if (args.Length > 1)
                    {
                        // Sprawdzamy, czy katalogi mają być sprawdzane rekurencyjnie.
                        if (args[0] == "/sub")
                        {
                            if (args.Length < 2)
                            {
                                ShowUsage();
                                return;
                            }
                            recurseIntoSubdirectories = true;
                            firstDirectoryIndex = 1;
                        }
                    }

                    // Pobranie listy katalogów z wiersza poleceń.
                    directoriesToSearch = args.Skip(firstDirectoryIndex);
                }
                List<FileNameGroup> filesGroupedByName =
                    InspectDirectories(recurseIntoSubdirectories, directoriesToSearch);

                DisplayMatches(filesGroupedByName);
                Console.ReadKey();
            }
            finally
            {
                if (testDirectoriesMade)
                {
                    CleanupTestDirectories(directoriesToSearch);
                }
            }
        }

        private static void ShowUsage()
        {
            Console.WriteLine("Odnajdywanie duplikatów plików.");
            Console.WriteLine("====================");
            Console.WriteLine(
                "Program poszukuje powtarzających się plików w jednym lub kilku katalogach.");
            Console.WriteLine();
            Console.WriteLine(
                "Sposób korzystania: findduplicatefiles [/sub] NazwaKatalogu [NazwaKatalogu] ...");
            Console.WriteLine("/sub - rekurencyjne przeszukiwanie podkatalogów.");
            Console.ReadKey();
        }

        private static List<FileNameGroup> InspectDirectories(
            bool recurseIntoSubdirectories,
            IEnumerable<string> directoriesToSearch)
        {
            var searchOption = recurseIntoSubdirectories ?
                SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;

            // Pobranie ścieżki dostępu do każdego pliku w każdym 
            // z przeszukiwanych katalogów
            var allFilePaths = from directory in directoriesToSearch
                               from file in Directory.GetFiles(directory, "*.*",
                                                                searchOption)
                               select file;

            // Pogrupowanie plików na podstawie lokalnej nazwy (czyli nazwy bez
            // ścieżki dostępu do katalogu, w jakim plik się znajduje) oraz utworzenie 
            // dla każdej nazwy pliku listy zawierającej szczegółowe informacje 
            // dotyczące każdego pliku o tej nazwie  
            var fileNameGroups = from filePath in allFilePaths
                                 let fileNameWithoutPath = Path.GetFileName(filePath)
                                 group filePath by fileNameWithoutPath into nameGroup
                                 select new FileNameGroup
                                 {
                                     FileNameWithoutPath = nameGroup.Key,
                                     FilesWithThisName =
                                      (from filePath in nameGroup
                                       let info = new FileInfo(filePath)
                                       select new FileDetails
                                       {
                                           FilePath = filePath,
                                           FileSize = info.Length
                                       }).ToList()
                                 };

            return fileNameGroups.ToList();
        }

        private static void DisplayMatches(
            IEnumerable<FileNameGroup> filesGroupedByName)
        {
            var groupsWithMoreThanOneFile = from nameGroup in filesGroupedByName
                                            where nameGroup.FilesWithThisName.Count > 1
                                            select nameGroup;

            foreach (var fileNameGroup in groupsWithMoreThanOneFile)
            {
                // Pogrupowanie odpowiadających sobie plików na podstawie ich wielkości
                // i wybranie tych, których dla danej wielkości jest więcej niż jeden 
                var matchesBySize = from file in fileNameGroup.FilesWithThisName
                                    group file by file.FileSize into sizeGroup
                                    where sizeGroup.Count() > 1
                                    select sizeGroup;

                foreach (var matchedBySize in matchesBySize)
                {
                    string fileNameAndSize = string.Format("{0} ({1} bajtów)",
                    fileNameGroup.FileNameWithoutPath, matchedBySize.Key);
                    WriteWithUnderlines(fileNameAndSize);
                    // Wyświetlenie każdego katalogu zawierającego dany plik
                    foreach (var file in matchedBySize)
                    {
                        Console.WriteLine(Path.GetDirectoryName(file.FilePath));
                    }
                    Console.WriteLine();
                }
            }
        }

        private static void WriteWithUnderlines(string text)
        {
            Console.WriteLine(text);
            Console.WriteLine(new string('-', text.Length));
        }

        // Listing 11-14. Tworzenie katalogów testowych.
        private static string[] MakeTestDirectories()
        {
            string localApplicationData = Path.Combine(
                Environment.GetFolderPath(
                    Environment.SpecialFolder.LocalApplicationData),
                @"Programming CSharp\FindDuplicates");

            // Tworzymy trzy katalogi testowe.
            var directories = new string[3];
            for (int i = 0; i < directories.Length; ++i)
            {
                string directory = Path.GetRandomFileName();
                // Łączymy dane lokalnej aplikacji z losowymi 
                // nazwami plików i katalogów...
                string fullPath = Path.Combine(localApplicationData, directory);
                // ...i tworzymy katalog.
                Directory.CreateDirectory(fullPath);
                directories[i] = fullPath;
                Console.WriteLine(fullPath);
            }
            return directories;
        }

        // Listing 11-17. Usuwanie katalogu.
        private static void CleanupTestDirectories(IEnumerable<string> directories)
        {
            foreach (var directory in directories)
            {
                Directory.Delete(directory);
            }
        }
    }
}
